define([
    'underscore',
    'backbone',
    'marionette',
    'moment',
    'App',
    'modules/appointments/appointments-module',
    'modules/appointments/appointments-radio',

    'text!modules/new-appointment-request/views/requests/template.html',

    'modules/new-appointment-request/views/requests/purpose/section-view',
    'modules/new-appointment-request/views/requests/express-care/section-view',
    'modules/new-appointment-request/views/requests/date-time/layout-view',
    'modules/new-appointment-request/views/requests/phone-contact/section-view',
    'modules/new-appointment-request/views/requests/message/section-view',
],
function (
    _,
    Backbone,
    Marionette,
    moment,
    app,
    module,
    radio,

    template,

    TypePurposeSectionView,
    ExpressCareReasonSectionView,
    DateTimeLayoutView,
    PhoneContactSectionView,
    NewMessageSectionView
) {
    'use strict';

    return Backbone.Marionette.View.extend({
        tagName: 'fieldset',
        className: 'request-details-section',
        template: _.template(template),
        regions: {
            typePurposeRegion: '.request-details-type-purpose-section',
            dateTimeRegion: '.request-details-date-time-section',
            phoneContactRegion: '.request-details-phone-contact-section',
            newMessageRegion: '.request-details-new-message-section',
        },
        initialize: function(options) {
            this.setModelProviderValues();
            this.validator = options.validator;
        },
        setModelProviderValues: function() {
            this.model.set({
                providerId: '0',
                providerName: '',
                providerOption: '',
            });
        },
        onRender: function() {
            if (this.model.has('typeOfCare')) {
                if (this.model.getCareId() === module.clinicalServicesConstants.EXPRESS_CARE_ID) {
                    this.showExpressCareReasonSectionView();
                    this.setSubmitParamsForExpressCare();
                } else {
                    this.showTypePurposeAndDateTimeSectionViews();
                }
                this.showPhoneContactSectionView();
                this.showNewMessageSectionView();
            }
        },
        setSubmitParamsForExpressCare: function() {
            var today = new Date();
            var dateString = moment(today).format('MM/DD/YYYY');
            var timeString = moment(today).format('A');
            this.model.set({
                optionDate1: dateString,
                optionTime1: timeString,
                purposeOfVisit: 'Express Care Request',
                visitType: 'Express Care',
            });
        },
        showTypePurposeAndDateTimeSectionViews: function() {
            var typePurposeRegion = this.getRegion('typePurposeRegion');
            var dateTimeRegion = this.getRegion('dateTimeRegion');

            if (!this.isDestroyed()) {
                typePurposeRegion.show(new TypePurposeSectionView({
                    model: this.model,
                    validator: this.validator,
                }));

                dateTimeRegion.show(new DateTimeLayoutView({model: this.model}));
            }
        },
        showExpressCareReasonSectionView: function() {
            $.when(this.fetchReasons()).then(function() {
                var typePurposeRegion = this.getRegion('typePurposeRegion');

                if (!this.isDestroyed()) {
                    typePurposeRegion.show(new ExpressCareReasonSectionView({
                        model: this.model,
                        validator: this.validator,
                        reasons: module.expressCareReasons,
                    }));
                }
            }.bind(this));
        },
        fetchReasons: function() {
            return module.expressCareReasons.fetch({url: app.resources.get('express-care-reasons').get('href')});
        },
        showPhoneContactSectionView: function() {
            var phoneContactRegion = this.getRegion('phoneContactRegion');

            if (!this.isDestroyed()) {
                phoneContactRegion.show(new PhoneContactSectionView({model: this.model}));
            }
        },
        showNewMessageSectionView: function() {
            var newMessageRegion = this.getRegion('newMessageRegion');

            if (!this.isDestroyed()) {
                newMessageRegion.show(new NewMessageSectionView({model: this.model}));
            }
        },
        templateContext: function() {
            return {
                EXPRESS_CARE_ID: module.clinicalServicesConstants.EXPRESS_CARE_ID,
                typeOfCare: this.model.getCareId(),
            };
        },
    });
}
);
